<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\IcaEnrollment;
use App\Models\IcaProgram;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class IcaAdminController extends Controller
{
    public function enrollments(Request $request)
    {
        $query = IcaEnrollment::query()->orderBy('created_at', 'desc');

        if ($request->has('payment_status') && $request->payment_status != '') {
            $query->where('payment_status', $request->payment_status);
        }

        if ($request->has('enrollment_status') && $request->enrollment_status != '') {
            $query->where('enrollment_status', $request->enrollment_status);
        }

        $enrollments = $query->paginate(20);

        return view('admin.ica.enrollments.index', compact('enrollments'));
    }

    public function showEnrollment($id)
    {
        $enrollment = IcaEnrollment::findOrFail($id);
        $programs = $enrollment->programs();
        
        return view('admin.ica.enrollments.show', compact('enrollment', 'programs'));
    }

    public function approvePayment(Request $request, $id)
    {
        $enrollment = IcaEnrollment::findOrFail($id);
        
        $enrollment->payment_status = 'paid';
        $enrollment->payment_verified_at = now();
        $enrollment->verified_by = Auth::id();
        $enrollment->admin_notes = $request->input('admin_notes');
        $enrollment->save();

        return redirect()->back()->with('success', 'Payment approved successfully!');
    }

    public function rejectPayment(Request $request, $id)
    {
        $enrollment = IcaEnrollment::findOrFail($id);
        
        $enrollment->payment_status = 'rejected';
        $enrollment->admin_notes = $request->input('admin_notes');
        $enrollment->save();

        return redirect()->back()->with('success', 'Payment rejected.');
    }

    public function approveEnrollment($id)
    {
        $enrollment = IcaEnrollment::findOrFail($id);
        
        if ($enrollment->payment_status !== 'paid') {
            return redirect()->back()->with('error', 'Cannot approve enrollment without confirmed payment.');
        }

        $enrollment->enrollment_status = 'approved';
        $enrollment->save();

        return redirect()->back()->with('success', 'Enrollment approved successfully!');
    }

    public function rejectEnrollment(Request $request, $id)
    {
        $enrollment = IcaEnrollment::findOrFail($id);
        
        $enrollment->enrollment_status = 'rejected';
        $enrollment->admin_notes = $request->input('admin_notes');
        $enrollment->save();

        return redirect()->back()->with('success', 'Enrollment rejected.');
    }

    public function programs()
    {
        $programs = IcaProgram::orderBy('category')->orderBy('program_name')->paginate(20);
        return view('admin.ica.programs.index', compact('programs'));
    }

    public function createProgram()
    {
        return view('admin.ica.programs.create');
    }

    public function storeProgram(Request $request)
    {
        $validated = $request->validate([
            'category' => 'required|string|max:255',
            'program_name' => 'required|string|max:255',
            'program_code' => 'required|string|max:50|unique:ica_programs',
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'requirements' => 'nullable|string',
            'benefits' => 'nullable|string',
            'certification' => 'nullable|string',
            'duration_weeks' => 'nullable|integer|min:1',
            'location' => 'nullable|string|max:255',
            'is_active' => 'boolean'
        ]);

        IcaProgram::create($validated);

        return redirect()->route('admin.ica.programs')->with('success', 'Program created successfully!');
    }

    public function editProgram($id)
    {
        $program = IcaProgram::findOrFail($id);
        return view('admin.ica.programs.edit', compact('program'));
    }

    public function updateProgram(Request $request, $id)
    {
        $program = IcaProgram::findOrFail($id);

        $validated = $request->validate([
            'category' => 'required|string|max:255',
            'program_name' => 'required|string|max:255',
            'program_code' => 'required|string|max:50|unique:ica_programs,program_code,' . $id,
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'requirements' => 'nullable|string',
            'benefits' => 'nullable|string',
            'certification' => 'nullable|string',
            'duration_weeks' => 'nullable|integer|min:1',
            'location' => 'nullable|string|max:255',
            'is_active' => 'boolean'
        ]);

        $program->update($validated);

        return redirect()->route('admin.ica.programs')->with('success', 'Program updated successfully!');
    }

    public function deleteProgram($id)
    {
        $program = IcaProgram::findOrFail($id);
        $program->delete();

        return redirect()->route('admin.ica.programs')->with('success', 'Program deleted successfully!');
    }
}
