@extends('layouts.default')
@push('title', 'Program Enrollment - Industry Connect Academy')
@push('meta')@endpush
@push('css')
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
<style>
    :root {
        --ica-navy: #11273f;
        --ica-navy-light: #1a3a5f;
        --ica-orange: #f2862d;
        --ica-orange-light: #ff9f4d;
        --primary-gradient: linear-gradient(135deg, #11273f 0%, #1a3a5f 100%);
        --accent-gradient: linear-gradient(135deg, #f2862d 0%, #ff9f4d 100%);
    }

    * {
        font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
    }

    body {
        background: linear-gradient(135deg, #f8f9fb 0%, #e8eaed 100%);
        min-height: 100vh;
    }

    .enrollment-container {
        padding: 60px 0;
    }

    .enrollment-card {
        background: white;
        border-radius: 16px;
        box-shadow: 0 12px 40px rgba(17, 39, 63, 0.12);
        overflow: hidden;
        max-width: 1200px;
        margin: 0 auto;
        border: 1px solid #e8eaed;
    }

    .enrollment-header {
        background: var(--primary-gradient);
        padding: 40px;
        color: white;
        position: relative;
        overflow: hidden;
    }

    .enrollment-header::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, #f2862d, #ff9f4d);
    }

    .enrollment-header h1 {
        font-size: 2.5rem;
        font-weight: 800;
        margin-bottom: 10px;
        text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
    }

    .enrollment-header p {
        font-size: 1.1rem;
        opacity: 0.95;
        margin-bottom: 0;
    }

    /* Progress Steps */
    .progress-steps {
        display: flex;
        justify-content: space-between;
        padding: 40px 60px;
        background: #f8f9fa;
        position: relative;
    }

    .progress-steps::before {
        content: '';
        position: absolute;
        top: 70px;
        left: 10%;
        right: 10%;
        height: 3px;
        background: #e0e0e0;
        z-index: 0;
    }

    .step {
        display: flex;
        flex-direction: column;
        align-items: center;
        position: relative;
        z-index: 1;
        flex: 1;
    }

    .step-circle {
        width: 60px;
        height: 60px;
        border-radius: 50%;
        background: white;
        border: 3px solid #e0e0e0;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 700;
        font-size: 1.3rem;
        color: #999;
        transition: all 0.3s;
        margin-bottom: 12px;
    }

    .step.active .step-circle {
        background: var(--primary-gradient);
        border-color: var(--ica-navy);
        color: white;
        transform: scale(1.1);
        box-shadow: 0 5px 20px rgba(17, 39, 63, 0.4);
    }

    .step.completed .step-circle {
        background: var(--ica-navy);
        border-color: var(--ica-navy);
        color: white;
    }

    .step-label {
        font-size: 0.9rem;
        font-weight: 600;
        color: #666;
        text-align: center;
    }

    .step.active .step-label {
        color: var(--ica-navy);
    }

    /* Form Content */
    .form-content {
        padding: 50px 60px;
    }

    .form-section {
        display: none;
    }

    .form-section.active {
        display: block;
        animation: fadeInUp 0.5s ease;
    }

    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(20px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .section-title {
        font-size: 1.8rem;
        font-weight: 700;
        color: #1a1a1a;
        margin-bottom: 10px;
        display: flex;
        align-items: center;
        gap: 12px;
    }

    .section-title i {
        color: var(--ica-orange);
        font-size: 1.5rem;
    }

    .section-subtitle {
        color: #666;
        margin-bottom: 35px;
        font-size: 1.05rem;
    }

    .form-group-elegant {
        margin-bottom: 28px;
    }

    .form-label-elegant {
        font-weight: 600;
        color: #333;
        margin-bottom: 10px;
        display: block;
        font-size: 0.95rem;
    }

    .form-label-elegant .required {
        color: #dc3545;
        margin-left: 3px;
    }

    .form-control-elegant {
        width: 100%;
        padding: 14px 18px;
        border: 2px solid #e0e0e0;
        border-radius: 12px;
        font-size: 1rem;
        transition: all 0.3s;
        background: #fafafa;
    }

    .form-control-elegant:focus {
        outline: none;
        border-color: var(--ica-orange);
        background: white;
        box-shadow: 0 0 0 4px rgba(242, 134, 45, 0.1);
    }

    .form-control-elegant:disabled,
    .form-control-elegant[readonly] {
        background: #f0f0f0;
        cursor: not-allowed;
    }

    /* User Info Card */
    .user-info-card {
        background: linear-gradient(135deg, #e8f5e9 0%, #c8e6c9 100%);
        border-radius: 16px;
        padding: 20px;
        margin-bottom: 30px;
        border-left: 4px solid var(--ica-orange);
    }

    .user-info-card .user-avatar {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        background: var(--primary-gradient);
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 1.5rem;
        font-weight: 700;
    }

    /* Program Selection */
    .program-category-elegant {
        margin-bottom: 35px;
    }

    .category-header {
        background: var(--primary-gradient);
        color: white;
        padding: 18px 24px;
        border-radius: 12px;
        margin-bottom: 15px;
        display: flex;
        align-items: center;
        gap: 15px;
        cursor: pointer;
        transition: all 0.3s;
    }

    .category-header:hover {
        transform: translateX(5px);
        box-shadow: 0 5px 20px rgba(0, 107, 63, 0.3);
    }

    .category-icon {
        width: 50px;
        height: 50px;
        background: rgba(255, 255, 255, 0.2);
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.5rem;
    }

    .category-title {
        flex: 1;
        font-weight: 700;
        font-size: 1.1rem;
    }

    .category-count {
        background: rgba(255, 255, 255, 0.3);
        padding: 5px 15px;
        border-radius: 20px;
        font-size: 0.9rem;
    }

    .program-checkbox-card {
        background: white;
        border: 2px solid #e0e0e0;
        border-radius: 12px;
        padding: 20px;
        margin-bottom: 12px;
        cursor: pointer;
        transition: all 0.3s;
        display: flex;
        align-items: center;
        gap: 15px;
    }

    .program-checkbox-card:hover {
        border-color: var(--ica-orange);
        background: rgba(242, 134, 45, 0.05);
        transform: translateX(5px);
    }

    .program-checkbox-card.selected {
        border-color: var(--ica-orange);
        background: rgba(242, 134, 45, 0.1);
        box-shadow: 0 5px 15px rgba(242, 134, 45, 0.15);
    }

    .program-checkbox {
        width: 24px;
        height: 24px;
        border: 2px solid #ccc;
        border-radius: 6px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s;
    }

    .program-checkbox-card.selected .program-checkbox {
        background: var(--ica-orange);
        border-color: var(--ica-orange);
    }

    .program-checkbox-card.selected .program-checkbox::after {
        content: '✓';
        color: white;
        font-weight: 700;
    }

    .program-details {
        flex: 1;
    }

    .program-name {
        font-weight: 600;
        color: #1a1a1a;
        margin-bottom: 5px;
    }

    .program-meta {
        display: flex;
        gap: 15px;
        font-size: 0.85rem;
        color: #666;
    }

    .program-price {
        font-size: 1.3rem;
        font-weight: 700;
        color: var(--ica-navy);
    }

    /* Payment Options */
    .payment-option-elegant {
        background: white;
        border: 3px solid #e0e0e0;
        border-radius: 16px;
        padding: 25px;
        cursor: pointer;
        transition: all 0.3s;
        text-align: center;
        height: 100%;
    }

    .payment-option-elegant:hover {
        border-color: var(--ica-orange);
        transform: translateY(-5px);
        box-shadow: 0 10px 30px rgba(242, 134, 45, 0.15);
    }

    .payment-option-elegant.active {
        border-color: var(--ica-orange);
        background: rgba(242, 134, 45, 0.1);
        box-shadow: 0 10px 30px rgba(242, 134, 45, 0.2);
    }

    .payment-icon {
        width: 80px;
        height: 80px;
        margin: 0 auto 20px;
        background: var(--primary-gradient);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 2.5rem;
        color: white;
    }

    .payment-option-elegant.active .payment-icon {
        animation: pulse 1.5s infinite;
    }

    @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.05); }
    }

    /* Navigation Buttons */
    .form-navigation {
        display: flex;
        justify-content: space-between;
        gap: 15px;
        margin-top: 40px;
        padding-top: 30px;
        border-top: 2px solid #f0f0f0;
    }

    .btn-elegant {
        padding: 15px 40px;
        border-radius: 12px;
        font-weight: 600;
        font-size: 1.05rem;
        border: none;
        cursor: pointer;
        transition: all 0.3s;
        display: inline-flex;
        align-items: center;
        gap: 10px;
    }

    .btn-primary-elegant {
        background: var(--primary-gradient);
        color: white;
        box-shadow: 0 5px 20px rgba(0, 107, 63, 0.3);
    }

    .btn-primary-elegant:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 30px rgba(0, 107, 63, 0.4);
    }

    .btn-secondary-elegant {
        background: #f0f0f0;
        color: #666;
    }

    .btn-secondary-elegant:hover {
        background: #e0e0e0;
    }

    .btn-elegant:disabled {
        opacity: 0.5;
        cursor: not-allowed;
    }

    /* Alert Messages */
    .alert-elegant {
        padding: 18px 24px;
        border-radius: 12px;
        margin-bottom: 25px;
        display: flex;
        align-items: center;
        gap: 15px;
        font-weight: 500;
    }

    .alert-elegant i {
        font-size: 1.5rem;
    }

    .alert-info-elegant {
        background: #e3f2fd;
        border-left: 4px solid #2196f3;
        color: #1565c0;
    }

    .alert-success-elegant {
        background: #e8f5e9;
        border-left: 4px solid #4caf50;
        color: #2e7d32;
    }

    .alert-warning-elegant {
        background: #fff3e0;
        border-left: 4px solid #ff9800;
        color: #e65100;
    }

    /* Radio Buttons */
    .radio-group-elegant {
        display: flex;
        gap: 20px;
        margin-top: 10px;
    }

    .radio-option-elegant {
        flex: 1;
        padding: 15px;
        border: 2px solid #e0e0e0;
        border-radius: 12px;
        cursor: pointer;
        transition: all 0.3s;
        text-align: center;
        font-weight: 600;
    }

    .radio-option-elegant:hover {
        border-color: var(--ica-orange);
        background: rgba(242, 134, 45, 0.05);
    }

    .radio-option-elegant.selected {
        border-color: var(--ica-orange);
        background: rgba(242, 134, 45, 0.1);
        color: var(--ica-orange);
    }

    /* File Upload */
    .file-upload-elegant {
        border: 3px dashed #e0e0e0;
        border-radius: 16px;
        padding: 40px;
        text-align: center;
        cursor: pointer;
        transition: all 0.3s;
        background: #fafafa;
    }

    .file-upload-elegant:hover {
        border-color: var(--ica-orange);
        background: rgba(242, 134, 45, 0.05);
    }

    .file-upload-icon {
        font-size: 3rem;
        color: var(--ica-orange);
        margin-bottom: 15px;
    }

    /* Mobile Responsive */
    @media (max-width: 768px) {
        .enrollment-header {
            padding: 30px 20px;
        }

        .enrollment-header h1 {
            font-size: 1.8rem;
        }

        .progress-steps {
            padding: 30px 20px;
            overflow-x: auto;
        }

        .step-circle {
            width: 50px;
            height: 50px;
            font-size: 1.1rem;
        }

        .step-label {
            font-size: 0.75rem;
        }

        .form-content {
            padding: 30px 20px;
        }

        .section-title {
            font-size: 1.4rem;
        }

        .form-navigation {
            flex-direction: column;
        }

        .btn-elegant {
            width: 100%;
            justify-content: center;
        }
    }

    /* Loading Spinner */
    .spinner {
        display: inline-block;
        width: 20px;
        height: 20px;
        border: 3px solid rgba(255,255,255,.3);
        border-radius: 50%;
        border-top-color: white;
        animation: spin 1s ease-in-out infinite;
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }
</style>
@endpush

@section('content')
<div class="enrollment-container">
    <div class="container">
        <div class="enrollment-card">
            <!-- Header -->
            <div class="enrollment-header">
                <h1><i class="fas fa-graduation-cap"></i> Program Enrollment</h1>
                <p>Industry Connect Academy - Professional Industrial Training</p>
            </div>

            <!-- Progress Steps -->
            <div class="progress-steps">
                <div class="step active" data-step="1">
                    <div class="step-circle">1</div>
                    <div class="step-label">Personal Info</div>
                </div>
                <div class="step" data-step="2">
                    <div class="step-circle">2</div>
                    <div class="step-label">Select Programs</div>
                </div>
                <div class="step" data-step="3">
                    <div class="step-circle">3</div>
                    <div class="step-label">Experience</div>
                </div>
                <div class="step" data-step="4">
                    <div class="step-circle">4</div>
                    <div class="step-label">Payment</div>
                </div>
                <div class="step" data-step="5">
                    <div class="step-circle">5</div>
                    <div class="step-label">Review</div>
                </div>
            </div>

            <!-- Form Content -->
            <form action="{{ route('ica.enrollment.store') }}" method="POST" enctype="multipart/form-data" id="enrollmentForm">
                @csrf
                <div class="form-content">
                    @if(session('success'))
                    <div class="alert-elegant alert-success-elegant">
                        <i class="fas fa-check-circle"></i>
                        <span>{{ session('success') }}</span>
                    </div>
                    @endif

                    @if(session('error'))
                    <div class="alert-elegant alert-warning-elegant">
                        <i class="fas fa-exclamation-triangle"></i>
                        <span>{{ session('error') }}</span>
                    </div>
                    @endif

                    <!-- Step 1: Personal Information -->
                    <div class="form-section active" data-section="1">
                        <div class="section-title">
                            <i class="fas fa-user-circle"></i>
                            Personal Information
                        </div>
                        <p class="section-subtitle">Let's start with your basic information</p>

                        <div class="user-info-card">
                            <div class="d-flex align-items-center gap-3">
                                <div class="user-avatar">
                                    {{ strtoupper(substr(auth()->user()->name, 0, 1)) }}
                                </div>
                                <div>
                                    <div class="fw-bold">Logged in as: {{ auth()->user()->name }}</div>
                                    <div class="text-muted small">{{ auth()->user()->email }}</div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group-elegant">
                                    <label class="form-label-elegant">
                                        First Name <span class="required">*</span>
                                    </label>
                                    <input type="text" name="first_name" class="form-control-elegant" 
                                           value="{{ old('first_name', explode(' ', auth()->user()->name)[0] ?? '') }}" 
                                           required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group-elegant">
                                    <label class="form-label-elegant">
                                        Surname <span class="required">*</span>
                                    </label>
                                    <input type="text" name="surname" class="form-control-elegant" 
                                           value="{{ old('surname', explode(' ', auth()->user()->name)[count(explode(' ', auth()->user()->name)) - 1] ?? '') }}" 
                                           required>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group-elegant">
                                    <label class="form-label-elegant">
                                        Email Address <span class="required">*</span>
                                    </label>
                                    <input type="email" name="email" class="form-control-elegant" 
                                           value="{{ old('email', auth()->user()->email) }}" 
                                           readonly required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group-elegant">
                                    <label class="form-label-elegant">
                                        Mobile Number <span class="required">*</span>
                                    </label>
                                    <input type="tel" name="mobile" class="form-control-elegant" 
                                           value="{{ old('mobile') }}" 
                                           placeholder="+233 XX XXX XXXX" required>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group-elegant">
                                    <label class="form-label-elegant">
                                        Age <span class="required">*</span>
                                    </label>
                                    <input type="number" name="age" class="form-control-elegant" 
                                           value="{{ old('age') }}" 
                                           min="18" max="100" required>
                                </div>
                            </div>
                            <div class="col-md-8">
                                <div class="form-group-elegant">
                                    <label class="form-label-elegant">
                                        City <span class="required">*</span>
                                    </label>
                                    <input type="text" name="city" class="form-control-elegant" 
                                           value="{{ old('city') }}" 
                                           placeholder="e.g., Accra, Kumasi, Tema" required>
                                </div>
                            </div>
                        </div>

                        <div class="form-group-elegant">
                            <label class="form-label-elegant">
                                Residential Address <span class="required">*</span>
                            </label>
                            <textarea name="address" class="form-control-elegant" rows="3" 
                                      placeholder="Enter your full residential address" required>{{ old('address') }}</textarea>
                        </div>

                        <div class="form-group-elegant">
                            <label class="form-label-elegant">
                                Educational Level <span class="required">*</span>
                            </label>
                            <select name="educational_level" class="form-control-elegant" required>
                                <option value="">Select your educational level</option>
                                <option value="SHS/TVET" {{ old('educational_level') == 'SHS/TVET' ? 'selected' : '' }}>SHS/TVET</option>
                                <option value="DIPLOMA/DEGREE" {{ old('educational_level') == 'DIPLOMA/DEGREE' ? 'selected' : '' }}>Diploma/Degree</option>
                                <option value="MASTERS" {{ old('educational_level') == 'MASTERS' ? 'selected' : '' }}>Masters</option>
                                <option value="OTHER" {{ old('educational_level') == 'OTHER' ? 'selected' : '' }}>Other</option>
                            </select>
                        </div>
                    </div>

                    <!-- Step 2: Program Selection -->
                    <div class="form-section" data-section="2">
                        <div class="section-title">
                            <i class="fas fa-clipboard-list"></i>
                            Select Training Programs
                        </div>
                        <p class="section-subtitle">Choose one or multiple programs that interest you</p>

                        <div class="alert-elegant alert-info-elegant">
                            <i class="fas fa-info-circle"></i>
                            <span>You can select multiple programs within the same category for a structured learning experience.</span>
                        </div>

                        @php
                            $programs = \App\Models\IcaProgram::active()->orderBy('category')->orderBy('program_name')->get();
                            $programsByCategory = $programs->groupBy('category');
                        @endphp

                        @foreach($programsByCategory as $category => $categoryPrograms)
                        <div class="program-category-elegant">
                            <div class="category-header" onclick="toggleCategory(this)">
                                <div class="category-icon">
                                    @if(str_contains($category, 'PROJECT'))
                                        <i class="fas fa-project-diagram"></i>
                                    @elseif(str_contains($category, 'HEIGHT'))
                                        <i class="fas fa-hard-hat"></i>
                                    @elseif(str_contains($category, 'CORROSION'))
                                        <i class="fas fa-spray-can"></i>
                                    @elseif(str_contains($category, 'NDT'))
                                        <i class="fas fa-microscope"></i>
                                    @elseif(str_contains($category, 'HSE'))
                                        <i class="fas fa-shield-alt"></i>
                                    @endif
                                </div>
                                <div class="category-title">{{ $category }}</div>
                                <div class="category-count">{{ $categoryPrograms->count() }} Programs</div>
                                <i class="fas fa-chevron-down"></i>
                            </div>

                            <div class="category-programs">
                                @foreach($categoryPrograms as $program)
                                <div class="program-checkbox-card" onclick="toggleProgram(this, '{{ $program->program_code }}')">
                                    <div class="program-checkbox"></div>
                                    <input type="checkbox" name="programs_of_interest[]" value="{{ $program->program_code }}" 
                                           id="program_{{ $program->id }}" style="display: none;">
                                    <div class="program-details">
                                        <div class="program-name">{{ $program->program_name }}</div>
                                        <div class="program-meta">
                                            <span><i class="fas fa-clock"></i> {{ $program->duration_weeks }} weeks</span>
                                            <span><i class="fas fa-map-marker-alt"></i> {{ $program->location }}</span>
                                        </div>
                                    </div>
                                    <div class="program-price">
                                        @if($program->price > 0)
                                            GHS {{ number_format($program->price, 0) }}
                                        @else
                                            <span class="badge bg-info">Contact Us</span>
                                        @endif
                                    </div>
                                </div>
                                @endforeach
                            </div>
                        </div>
                        @endforeach

                        <div id="program-error" class="text-danger mt-3" style="display: none;">
                            <i class="fas fa-exclamation-circle"></i> Please select at least one program
                        </div>
                    </div>

                    <!-- Step 3: Work Experience -->
                    <div class="form-section" data-section="3">
                        <div class="section-title">
                            <i class="fas fa-briefcase"></i>
                            Work Experience
                        </div>
                        <p class="section-subtitle">Tell us about your professional background</p>

                        <div class="form-group-elegant">
                            <label class="form-label-elegant">
                                Do you have previous experience in the programs you selected? <span class="required">*</span>
                            </label>
                            <div class="radio-group-elegant">
                                <div class="radio-option-elegant" onclick="selectRadio(this, 'has_previous_experience', '1')">
                                    <i class="fas fa-check-circle"></i> Yes, I have experience
                                    <input type="radio" name="has_previous_experience" value="1" style="display: none;" required>
                                </div>
                                <div class="radio-option-elegant" onclick="selectRadio(this, 'has_previous_experience', '0')">
                                    <i class="fas fa-times-circle"></i> No, I'm a beginner
                                    <input type="radio" name="has_previous_experience" value="0" style="display: none;" required>
                                </div>
                            </div>
                        </div>

                        <div class="form-group-elegant">
                            <label class="form-label-elegant" id="experienceLabel">
                                Describe Your Work Experience <span class="required">*</span>
                            </label>
                            <textarea name="work_experience" id="workExperienceField" class="form-control-elegant" rows="6" 
                                      placeholder="Please describe your relevant work experience, including job titles, companies, duration, and responsibilities." 
                                      required>{{ old('work_experience') }}</textarea>
                            <small class="text-muted" id="experienceHint">Minimum 50 characters</small>
                        </div>
                    </div>

                    <!-- Step 4: Payment Method -->
                    <div class="form-section" data-section="4">
                        <div class="section-title">
                            <i class="fas fa-credit-card"></i>
                            Payment Method
                        </div>
                        <p class="section-subtitle">Choose how you'd like to pay the enrollment fee</p>

                        <div class="alert-elegant alert-info-elegant">
                            <i class="fas fa-money-bill-wave"></i>
                            <span><strong>Enrollment Fee: GHS 100.00</strong> (Expression of Interest)</span>
                        </div>

                        <div class="row g-4">
                            <div class="col-md-4">
                                <div class="payment-option-elegant" onclick="selectPayment(this, 'paystack')">
                                    <input type="radio" name="payment_method" value="paystack" style="display: none;" required>
                                    <div class="payment-icon">
                                        <i class="fas fa-mobile-alt"></i>
                                    </div>
                                    <h5 class="fw-bold mb-2">Mobile Money</h5>
                                    <p class="text-muted small mb-0">Pay instantly via Paystack</p>
                                    <p class="text-muted small">MTN • Vodafone • AirtelTigo</p>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="payment-option-elegant" onclick="selectPayment(this, 'bank_transfer')">
                                    <input type="radio" name="payment_method" value="bank_transfer" style="display: none;" required>
                                    <div class="payment-icon">
                                        <i class="fas fa-university"></i>
                                    </div>
                                    <h5 class="fw-bold mb-2">Bank Transfer</h5>
                                    <p class="text-muted small mb-0">Upload bank receipt</p>
                                    <p class="text-muted small">Manual verification</p>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="payment-option-elegant" onclick="selectPayment(this, 'mobile_money')">
                                    <input type="radio" name="payment_method" value="mobile_money" style="display: none;" required>
                                    <div class="payment-icon">
                                        <i class="fas fa-money-bill-wave"></i>
                                    </div>
                                    <h5 class="fw-bold mb-2">Direct Mobile Money</h5>
                                    <p class="text-muted small mb-0">Upload transaction screenshot</p>
                                    <p class="text-muted small">Manual verification</p>
                                </div>
                            </div>
                        </div>

                        <div id="receiptUpload" style="display: none; margin-top: 30px;">
                            <div class="alert-elegant alert-warning-elegant">
                                <i class="fas fa-info-circle"></i>
                                <div>
                                    <strong>Bank Transfer Details:</strong><br>
                                    <strong>Bank:</strong> [Bank Name] | <strong>Account:</strong> [Account Number]<br>
                                    <strong>Account Name:</strong> Industry Connect Academy<br>
                                    <strong>Mobile Money:</strong> [Mobile Money Number]
                                </div>
                            </div>

                            <div class="form-group-elegant">
                                <label class="form-label-elegant">
                                    Upload Payment Receipt/Screenshot <span class="required">*</span>
                                </label>
                                <div class="file-upload-elegant" onclick="document.getElementById('bank_receipt').click()">
                                    <div class="file-upload-icon">
                                        <i class="fas fa-cloud-upload-alt"></i>
                                    </div>
                                    <h5 class="fw-bold mb-2">Click to Upload</h5>
                                    <p class="text-muted mb-0">JPG, PNG, or PDF (Max 5MB)</p>
                                    <input type="file" name="bank_receipt" id="bank_receipt" accept=".jpg,.jpeg,.png,.pdf" style="display: none;" onchange="displayFileName(this)">
                                </div>
                                <div id="fileName" class="mt-2 text-success fw-bold" style="display: none;"></div>
                            </div>
                        </div>
                    </div>

                    <!-- Step 5: Review & Submit -->
                    <div class="form-section" data-section="5">
                        <div class="section-title">
                            <i class="fas fa-check-circle"></i>
                            Review Your Application
                        </div>
                        <p class="section-subtitle">Please review your information before submitting</p>

                        <div class="alert-elegant alert-success-elegant">
                            <i class="fas fa-info-circle"></i>
                            <span>Once submitted, you will receive a confirmation email with next steps.</span>
                        </div>

                        <div id="reviewContent" class="mt-4">
                            <!-- Review content will be populated by JavaScript -->
                        </div>

                        <div class="form-check mt-4">
                            <input class="form-check-input" type="checkbox" id="termsAgree" required>
                            <label class="form-check-label fw-bold" for="termsAgree">
                                I confirm that all information provided is accurate and I agree to the terms and conditions
                            </label>
                        </div>
                    </div>

                    <!-- Navigation Buttons -->
                    <div class="form-navigation">
                        <button type="button" class="btn-elegant btn-secondary-elegant" id="prevBtn" onclick="changeStep(-1)" style="display: none;">
                            <i class="fas fa-arrow-left"></i> Previous
                        </button>
                        <div style="flex: 1;"></div>
                        <button type="button" class="btn-elegant btn-primary-elegant" id="nextBtn" onclick="changeStep(1)">
                            Next <i class="fas fa-arrow-right"></i>
                        </button>
                        <button type="submit" class="btn-elegant btn-primary-elegant" id="submitBtn" style="display: none;">
                            <i class="fas fa-paper-plane"></i> Submit Enrollment
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
let currentStep = 1;
const totalSteps = 5;

function changeStep(direction) {
    // Validate current step before moving forward
    if (direction === 1 && !validateStep(currentStep)) {
        return;
    }

    // Update step
    const newStep = currentStep + direction;
    if (newStep < 1 || newStep > totalSteps) return;

    // Hide current section
    document.querySelector(`.form-section[data-section="${currentStep}"]`).classList.remove('active');
    document.querySelector(`.step[data-step="${currentStep}"]`).classList.remove('active');
    if (direction === 1) {
        document.querySelector(`.step[data-step="${currentStep}"]`).classList.add('completed');
    }

    // Show new section
    currentStep = newStep;
    document.querySelector(`.form-section[data-section="${currentStep}"]`).classList.add('active');
    document.querySelector(`.step[data-step="${currentStep}"]`).classList.add('active');

    // Update buttons
    document.getElementById('prevBtn').style.display = currentStep === 1 ? 'none' : 'inline-flex';
    document.getElementById('nextBtn').style.display = currentStep === totalSteps ? 'none' : 'inline-flex';
    document.getElementById('submitBtn').style.display = currentStep === totalSteps ? 'inline-flex' : 'none';

    // Populate review on last step
    if (currentStep === totalSteps) {
        populateReview();
    }

    // Scroll to top
    window.scrollTo({ top: 0, behavior: 'smooth' });
}

function validateStep(step) {
    if (step === 1) {
        // Validate personal information
        const requiredFields = ['first_name', 'surname', 'email', 'mobile', 'age', 'city', 'address', 'educational_level'];
        for (let field of requiredFields) {
            const input = document.querySelector(`[name="${field}"]`);
            if (!input.value.trim()) {
                alert(`Please fill in the ${field.replace('_', ' ')} field`);
                input.focus();
                return false;
            }
        }
    } else if (step === 2) {
        // Validate program selection
        const selected = document.querySelectorAll('input[name="programs_of_interest[]"]:checked');
        if (selected.length === 0) {
            document.getElementById('program-error').style.display = 'block';
            return false;
        }
        document.getElementById('program-error').style.display = 'none';
    } else if (step === 3) {
        // Validate work experience
        const experience = document.querySelector('[name="has_previous_experience"]:checked');
        const description = document.querySelector('[name="work_experience"]');
        if (!experience) {
            alert('Please indicate if you have previous experience');
            return false;
        }
        // Only validate description length if user has experience
        if (experience.value === '1') {
            if (!description.value.trim() || description.value.trim().length < 50) {
                alert('Please provide a detailed work experience description (minimum 50 characters)');
                description.focus();
                return false;
            }
        } else {
            // For beginners, if field is empty, set a default value
            if (!description.value.trim()) {
                description.value = 'No previous experience';
            }
        }
    } else if (step === 4) {
        // Validate payment method
        const payment = document.querySelector('[name="payment_method"]:checked');
        if (!payment) {
            alert('Please select a payment method');
            return false;
        }
        if ((payment.value === 'bank_transfer' || payment.value === 'mobile_money')) {
            const receipt = document.getElementById('bank_receipt');
            if (!receipt.files.length) {
                alert('Please upload your payment receipt');
                return false;
            }
        }
    }
    return true;
}

function toggleCategory(element) {
    const programs = element.nextElementSibling;
    const icon = element.querySelector('.fa-chevron-down');
    
    if (programs.style.display === 'none') {
        programs.style.display = 'block';
        icon.style.transform = 'rotate(180deg)';
    } else {
        programs.style.display = 'none';
        icon.style.transform = 'rotate(0deg)';
    }
}

function toggleProgram(card, code) {
    const checkbox = card.querySelector('input[type="checkbox"]');
    checkbox.checked = !checkbox.checked;
    
    if (checkbox.checked) {
        card.classList.add('selected');
    } else {
        card.classList.remove('selected');
    }
}

function selectRadio(element, name, value) {
    // Remove selected class from all options
    element.parentElement.querySelectorAll('.radio-option-elegant').forEach(opt => {
        opt.classList.remove('selected');
    });
    
    // Add selected class to clicked option
    element.classList.add('selected');
    
    // Check the radio button
    const radio = element.querySelector('input[type="radio"]');
    radio.checked = true;
    
    // Update work experience field based on selection
    if (name === 'has_previous_experience') {
        const workExpField = document.getElementById('workExperienceField');
        const expLabel = document.getElementById('experienceLabel');
        const expHint = document.getElementById('experienceHint');
        
        if (value === '0') {
            // No experience - make field optional and update text
            workExpField.required = false;
            workExpField.placeholder = "Optional: Describe any related interests, hobbies, or reasons for wanting to join this program. You may also write 'No previous experience'.";
            expLabel.innerHTML = 'Additional Information <span class="text-muted">(Optional)</span>';
            expHint.textContent = 'This field is optional for beginners';
        } else {
            // Has experience - make field required
            workExpField.required = true;
            workExpField.placeholder = "Please describe your relevant work experience, including job titles, companies, duration, and responsibilities.";
            expLabel.innerHTML = 'Describe Your Work Experience <span class="required">*</span>';
            expHint.textContent = 'Minimum 50 characters';
        }
    }
}

function selectPayment(element, method) {
    // Remove active class from all options
    document.querySelectorAll('.payment-option-elegant').forEach(opt => {
        opt.classList.remove('active');
    });
    
    // Add active class to selected option
    element.classList.add('active');
    
    // Check the radio button
    const radio = element.querySelector('input[type="radio"]');
    radio.checked = true;
    
    // Show/hide receipt upload
    const receiptUpload = document.getElementById('receiptUpload');
    const receiptInput = document.getElementById('bank_receipt');
    
    if (method === 'bank_transfer' || method === 'mobile_money') {
        receiptUpload.style.display = 'block';
        receiptInput.required = true;
    } else {
        receiptUpload.style.display = 'none';
        receiptInput.required = false;
    }
}

function displayFileName(input) {
    const fileName = document.getElementById('fileName');
    if (input.files.length > 0) {
        fileName.textContent = '✓ ' + input.files[0].name;
        fileName.style.display = 'block';
    }
}

function populateReview() {
    const formData = new FormData(document.getElementById('enrollmentForm'));
    let html = '<div class="row g-4">';
    
    // Personal Information
    html += '<div class="col-md-6"><div class="card p-3"><h6 class="fw-bold mb-3"><i class="fas fa-user text-success"></i> Personal Information</h6>';
    html += `<p><strong>Name:</strong> ${formData.get('first_name')} ${formData.get('surname')}</p>`;
    html += `<p><strong>Email:</strong> ${formData.get('email')}</p>`;
    html += `<p><strong>Mobile:</strong> ${formData.get('mobile')}</p>`;
    html += `<p><strong>Age:</strong> ${formData.get('age')}</p>`;
    html += `<p><strong>City:</strong> ${formData.get('city')}</p>`;
    html += `<p><strong>Education:</strong> ${formData.get('educational_level')}</p>`;
    html += '</div></div>';
    
    // Selected Programs
    const programs = formData.getAll('programs_of_interest[]');
    html += '<div class="col-md-6"><div class="card p-3"><h6 class="fw-bold mb-3"><i class="fas fa-clipboard-list text-success"></i> Selected Programs</h6>';
    html += `<p><strong>${programs.length} program(s) selected</strong></p>`;
    programs.forEach(code => {
        const card = document.querySelector(`input[value="${code}"]`).closest('.program-checkbox-card');
        const name = card.querySelector('.program-name').textContent;
        html += `<p class="small">• ${name}</p>`;
    });
    html += '</div></div>';
    
    // Work Experience
    html += '<div class="col-12"><div class="card p-3"><h6 class="fw-bold mb-3"><i class="fas fa-briefcase text-success"></i> Work Experience</h6>';
    const hasExp = formData.get('has_previous_experience') === '1' ? 'Yes' : 'No';
    html += `<p><strong>Previous Experience:</strong> ${hasExp}</p>`;
    html += `<p><strong>Description:</strong> ${formData.get('work_experience').substring(0, 200)}...</p>`;
    html += '</div></div>';
    
    // Payment Method
    html += '<div class="col-12"><div class="card p-3"><h6 class="fw-bold mb-3"><i class="fas fa-credit-card text-success"></i> Payment Information</h6>';
    const payment = formData.get('payment_method');
    const paymentNames = {
        'paystack': 'Mobile Money (Paystack)',
        'bank_transfer': 'Bank Transfer',
        'mobile_money': 'Direct Mobile Money'
    };
    html += `<p><strong>Payment Method:</strong> ${paymentNames[payment]}</p>`;
    html += `<p><strong>Enrollment Fee:</strong> GHS 100.00</p>`;
    html += '</div></div>';
    
    html += '</div>';
    
    document.getElementById('reviewContent').innerHTML = html;
}

// Form submission
document.getElementById('enrollmentForm').addEventListener('submit', function(e) {
    const submitBtn = document.getElementById('submitBtn');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<span class="spinner"></span> Processing...';
});

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    // Collapse all categories by default except first
    document.querySelectorAll('.category-programs').forEach((programs, index) => {
        if (index > 0) {
            programs.style.display = 'none';
        }
    });
});
</script>
@endsection
